package javax.ide.extension.spi;

import java.net.URI;

import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;

import javax.ide.extension.DeferredElementVisitorHook;
import javax.ide.extension.ElementContext;
import javax.ide.extension.ElementEndContext;
import javax.ide.extension.ElementStartContext;
import javax.ide.extension.ElementVisitor;
import javax.ide.extension.Extension;
import javax.ide.util.MetaResourceString;

/**
 * A minimal visitor which just processes the id, version, esdk-version, name and
 * dependencies of an extension.
 */
public class MinimalExtensionVisitor extends BaseExtensionVisitor
{
  private final Map<Extension, ExtensionSource> _sourcesByExtension = new LinkedHashMap<Extension, ExtensionSource>();
  private final Map<String, Extension> _idToMinimalExtensionMap = new HashMap<String, Extension>();

  public Map<Extension, ExtensionSource> getSourcesByExtension()
  {
    return _sourcesByExtension;
  }

  public Map<String, Extension> getIdToMinimalExtensionMap()
  {
    return _idToMinimalExtensionMap;
  }

  @Override
  public void start(ElementStartContext start)
  {
    MinimalExtension ext = (MinimalExtension) processExtension(start);
    if (ext != null)
    {
      start.registerChildVisitor(NAME_ELEMENT, ext.getNameVisitor());
    }
  }

  @Override
  public void extension(ElementContext context, Extension extension)
  {
    ExtensionSource source = (ExtensionSource) context.getScopeData().get(ExtensionVisitor.KEY_EXTENSION_SOURCE);
    if (extension != null)
    {
      _idToMinimalExtensionMap.put(extension.getID(), extension);
      _sourcesByExtension.put(extension, source);
    }
  }

  @Override
  public void addToClasspath(ElementContext context, Extension extension, URI classpathEntry)
  {
    // NO-OP
  }

  @Override
  protected DefaultExtension createExtension(String id)
  {
    return new MinimalExtension(id);
  }

  protected class MinimalExtension extends DefaultExtension
  {
    public MinimalExtension(String id)
    {
      super(id);
    }

    public ElementVisitor getNameVisitor()
    {
      return _nameVisitor;
    }

    @Override
    public String getName()
    {
      //We go through the hoop of using DeferredElementVisitor in order to defer 
      //the loading of the resource bundle until this method is called (we don't
      //want to load bundles from all extension sources on startup, particularily
      //because this may never be called in a given execution of the IDE)
      String name = super.getName();
      if (name == null)
      {
        //there is a lock inside of attachElementVisitor that does the
        //right thing if called by two threads with the same visitor.
        //The second thread will wait until the first thread is done
        //replaying the stored data, and then be a no-op
        _nameVisitor.attachElementVisitor(_realNameVisitor);
      }
      return super.getName();
    }

    private DeferredElementVisitorHook _nameVisitor = new DeferredElementVisitorHook();
    private MinimalNameVisitor _realNameVisitor = new MinimalNameVisitor();
  }

  private class MinimalNameVisitor extends ElementVisitor
  {
    public void start(ElementStartContext context)
    {
      String rsKey = getAttributeHelper(context, "rskey", true, false);
      context.getScopeData().put("rskey", rsKey);
    }

    public void end(ElementEndContext context)
    {
      DefaultExtension extension = getExtension(context);
      DefaultElementContext dec = (DefaultElementContext) context;
      String rsKey = (String) context.getScopeData().get("rskey");
      String rawName = null;

      if (rsKey != null)
      {
        rawName = "${" + rsKey + "}";
      }
      else
      {
        rawName = dec.getRawText();
      }
      
      if (rawName != null)
      {
        extension.setRawName(rawName);
        extension.setName(dec.processText(rawName));
      }
      
      if (MacroExpander.containsMacro(rawName))
      {
        extension.setName( new MetaResourceString(getResourceBundle(context),
                                                  MacroExpander.stripOffBrackets(rawName)) );
      }
      else
      {
        extension.setName(rawName);   
      }
    }
  }
}
