/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.build;

import java.util.ArrayList;

import java.util.Collection;
import java.util.Collections;
import java.util.List;

import javax.ide.Service;
import javax.ide.spi.ProviderNotFoundException;
import javax.ide.build.spi.BuildSystemHook;
import javax.ide.command.Context;
import javax.ide.extension.ExtensionRegistry;

/**
 * The <code>BuildSystem</code> service allows extension writers to 
 * initiate a build and query information from the build system.
 */
public abstract class BuildSystem extends Service
{

  /*
   * A list of build listener instances. 
   */
  private final List _listeners = new ArrayList();
  

  protected BuildSystem()
  {
    
  }
  
  /**
   * Get whether there are any registered listeners.
   * 
   * @return true if there are any registered listeners.
   */
  protected final boolean hasListeners()
  {
    return !_listeners.isEmpty();
  }
  
  /**
   * Get all listeners. This method returns an immutable copy of the listener
   * list, so it's safe to iterate this and fire events directly.
   * 
   * @return all listeners.
   */
  protected final Collection getListeners()
  {
    return Collections.unmodifiableList( new ArrayList( _listeners ) );
  }
  
  /**
   * Initializes the build system manager.
   * <p>
   * This implementation retrieves and registers listeners obtained from
   * the build system hook handler.
   */
  protected void initialize()
  {
    BuildSystemHook hook = (BuildSystemHook)
      ExtensionRegistry.getExtensionRegistry().getHook( BuildSystemHook.ELEMENT );
    _listeners.addAll( hook.getListeners() );
  }

  /**
   * Add an {@link javax.ide.build.BuildListener}.
   *
   * @param l the {@link javax.ide.build.BuildListener} to add.
   */
  public final void addBuildListener( BuildListener l )
  {
    _listeners.add( l );
  }

  /**
   * Remove an {@link javax.ide.build.BuildListener}.
   *
   * @param l The {@link javax.ide.build.BuildListener} to remove. 
   */
  public final void removeBuildListener( BuildListener l )
  {
    _listeners.remove( l );
  }

  /**
   * Start a build of the objects selected in the {@link Context}. This 
   * method starts the build in another thread and returns immediately.
   *
   * @param context the objects returned by calling the
   * {@link Context#getSelection()} method are build using the classpath 
   * and source path information provided by the context project.
   */
  public abstract void build( Context context );

  /**
   * Check if the build system is currently building something. 
   *
   * @return <code>true</code> is a build is in progress.
   */
  public abstract boolean isBuilding();
  

  /**
   * Gets the build system implementation.
   * 
   * @return the build system implementation for this IDE. Must not return null.
   */
  public static BuildSystem getBuildSystem()
  {
    try
    {
      return (BuildSystem) getService( BuildSystem.class );
    }
    catch ( ProviderNotFoundException nse )
    {
      nse.printStackTrace();
      throw new IllegalStateException( "No build system service." );
    }
  }
}
