/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.extension;

import java.util.Collection;
import javax.ide.Identifiable;
import javax.ide.net.URIPath;
import javax.ide.util.Version;
 
/**
 * An extension is a unit of functionality in a JSR-198 based IDE. The
 * <tt>Extension</tt> interface provides access to metadata about an extension
 * provided in its extension manifest.
 */
public interface Extension extends Identifiable
{
  //---------------------------------------------------------------------------
  // Public methods.
  //---------------------------------------------------------------------------

  /**
   * Returns the name of this extension. This corresponds to the &lt;name>
   * element in the extension manifest.
   *
   * @return The name of this extension.
   */
  public String getName();

  /**
   * Returns the owner of this extension. This corresponds to the &lt;owner>
   * element in the extension manifest.
   *
   * @return The owner of this extension.
   */
  public String getOwner();

  /**
   * Returns the version of this extension. This corresponds to the 
   * version attribute.
   *
   * @return The version number of this extension.
   */
  public Version getVersion();

  /**
   * Returns the version of the Extension Software Development Kit (ESDK) 
   * this extension is compatible with. This corresponds to the 
   * esdk-version attribute on &lt;extension> in the extension manifest.<p>
   *
   * A JSR-198 compatible IDE may elect not to load an extension if the
   * IDE does not support the specified ESDK version.
   *
   * @return The version of the ESDK this extension is compatible with.
   */
  public Version getEDKVersion();

  /**
   * Returns information about the platforms on which this extension is
   * supported.<p>
   *
   * A JSR-198 compatible IDE may elect not to load an extension if the
   * IDE does not support the platforms required by the extension.
   *
   * @return information about the platforms this extension requires.
   */
  public PlatformInfo getPlatformInfo();

  /**
   * Returns a collection of {@link ExtensionDependency} objects, one for each 
   * dependency this extension has on another extension. The information 
   * here corresponds to the &lt;dependencies> section of the extension
   * manifest.
   * 
   * @return a collection of {@link ExtensionDependency} objects. May be empty.
   */
  public Collection<ExtensionDependency> getDependencies();

  /**
   * Returns the class path of this extension. The information here 
   * corresponds to the &lt;classpaths> section of the extension manifest.
   *
   * @return a URIPath containing classpath entries specified by this
   *   extension.
   */
  public URIPath getClassPath();
}
