/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.extension;
import java.util.ArrayList;
import java.util.Collection;

/**
 * Encapsulates platform specific information required for an extension to
 * work properly.
 */
public final class PlatformInfo
{
  /**
   * GUI toolkit identifier for Swing.
   */
  public static final String SWING = "swing";

  /**
   * GUI toolkit identifier for SWT.
   */
  public static final String SWT = "swt";

  private final Collection _toolkits;
  
  /**
   * Create a <tt>PlatformInfo</tt> object for the specified toolkits.
   * 
   * @param toolkits all toolkits supported by this extension. The first toolkit
   *    in the collection should be the preferred toolkit. Must not be null
   *    or empty.
   */
  public PlatformInfo( Collection toolkits )
  {
    if ( toolkits == null )
    {
      throw new NullPointerException( "Null toolkits" ); 
    }
    if ( toolkits.isEmpty() )
    {
      throw new IllegalArgumentException( "Empty toolkits" );
    }
    // Clone for safety.
    _toolkits = new ArrayList( toolkits );
  }

  /**
   * Get the preferred toolkit for this extension.
   * 
   * @return the preferred toolkit for this extension.
   */
  public String getPreferredToolkit()
  {
    return (String)_toolkits.iterator().next();
  }


  /**
   * Gets whether a specific toolkit is supported.
   *
   * @param toolkit to the toolkit to check for support. Either SWING or SWT.
   * @return true if the specified toolkit is supported.
   */
  public boolean isToolkitSupported( String toolkit )
  {
    return _toolkits.contains( toolkit );
  }
}
