/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.extension.spi;

import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.Map;

import javax.ide.extension.ElementContext;
import javax.ide.extension.ElementEndContext;
import javax.ide.extension.ElementName;
import javax.ide.extension.ElementStartContext;
import javax.ide.extension.ElementVisitor;
import javax.ide.extension.ElementVisitorFactory;
import javax.ide.extension.Extension;
import javax.ide.extension.ExtensionHook;
import javax.ide.util.MetaResourceString;


/**
 * Visitor for the root JSR-198 extension element. This is the "entry point"
 * for all manifest processing.
 */
public abstract class ExtensionVisitor extends BaseExtensionVisitor
{

  private static final ElementName CLASSPATHS = 
    new ElementName( ExtensionHook.MANIFEST_XMLNS, "classpaths" );

  private ElementVisitor _nameVisitor = createNameVisitor();
  private ElementVisitor _ownerVisitor = createOwnerVisitor();
  private ElementVisitor _classpathsVisitor = createClasspathsVisitor();
  private ElementVisitor _dependenciesVisitor = createDependenciesVisitor();
 

  /**
   * The key for the <tt>ClassLoader</tt> to be used to when looking up classes
   * for the current extension. This is used by I18NStringVisitor / 
   * I18NCharVisitor. If no classloader is in the scope map, the context
   * classloader of the current thread is used.
   */
  public static final String KEY_CLASSLOADER = "classLoader";
  
  private static final String KEY_LAST_RSKEY = "lastRSKey";

  private Map<String,Extension> _extensionsById = new LinkedHashMap<String,Extension>();
  
  
  protected ExtensionVisitor( ElementVisitorFactory hookFactory )
  {
    setHookVisitorFactory( hookFactory );
  }
  
  /**
   * Returns the extension with the specified id.
   * 
   * @param id the id to look up
   * @return the extension with the specified id, or null.
   * @since 2.0
   */
  public final Extension findExtension( String id )
  {
    return _extensionsById.get( id );
  }
  
  public final Collection<Extension> getExtensions()
  {
    return _extensionsById.values();
  }
    
  public final void start( ElementStartContext context )
  {
    Extension ext = processExtension( context );
    if ( ext == null )
    {
      return;
    }
    
    _extensionsById.put( ext.getID(), ext );
    context.getScopeData().put( ExtensionVisitor.KEY_CLASSLOADER, 
      getClassLoader( ext ) );
    
    String rsbundleClass = getAttributeHelper( context, ExtensionHook.ATTRIBUTE_RSBUNDLE_CLASS, true, false );
    if ( rsbundleClass != null )
    {
      context.getScopeData().put( ExtensionHook.KEY_RSBUNDLE_CLASS, rsbundleClass );
    }
    
    context.registerChildVisitor( NAME_ELEMENT, _nameVisitor );
    context.registerChildVisitor( OWNER_ELEMENT, _ownerVisitor );
    
    // With OSGi the classpath visitor and the dependencies visitors are both ignored. Allow for null here.
    if (_classpathsVisitor != null)
    {
      context.registerChildVisitor( CLASSPATHS, _classpathsVisitor );
    }
    if (_dependenciesVisitor != null)
    {
      context.registerChildVisitor( DependenciesVisitor.ELEMENT, 
        _dependenciesVisitor );
    }
  }
    
  protected ElementVisitor createClasspathsVisitor()
  {
    return new ClasspathsVisitor();
  }
  
  protected ElementVisitor createDependenciesVisitor()
  {
    return new DependenciesVisitor();
  }

  /**
   * A visitor for the name element that will set the raw name on the Extension
   */
  protected ElementVisitor createNameVisitor()
  {
    return new ElementVisitor()
    {
      public void start( ElementStartContext context )
      {
        String rsKey = context.getAttributeValue( "rskey" );
        if ( rsKey == "" ) rsKey = null;
        context.getScopeData().put( KEY_LAST_RSKEY, rsKey );
      }
    
      public void end( ElementEndContext context )
      {
        DefaultExtension extension = getExtension( context );
      
        String rsKey = (String) context.getScopeData().get( KEY_LAST_RSKEY );
        if ( rsKey != null )
        {
          extension.setRawName( "${" + rsKey + "}" );
        }
        else
        {
          DefaultElementContext dec = (DefaultElementContext) context;  
          extension.setRawName( dec.getRawText() );
        }
        
        final String rawName = extension.getRawName();
        if (MacroExpander.containsMacro(rawName))
        {
          extension.setName( new MetaResourceString(getResourceBundle(context), 
                                                    MacroExpander.stripOffBrackets(rawName)) );
        }
        else 
        {
          extension.setName(rawName);
        }
      }
    };
  }  

  /**
   * A visitor for the owner element that will set the raw owner on the Extension
   */
  protected ElementVisitor createOwnerVisitor()
  {
    return new ElementVisitor()
    {
      public void start( ElementStartContext context )
      {
        String rsKey = context.getAttributeValue( "rskey" );
        if ( rsKey == "" ) rsKey = null;
        context.getScopeData().put( KEY_LAST_RSKEY, rsKey );
      }
      
      public void end( ElementEndContext context )
      {
        DefaultExtension extension = getExtension( context );
      
        String rsKey = (String) context.getScopeData().get( KEY_LAST_RSKEY );
        if ( rsKey != null )
        {
          extension.setRawOwner( "${" + rsKey + "}" );
        }
        else
        {
          DefaultElementContext dec = (DefaultElementContext) context;  
          extension.setRawOwner( dec.getRawText() );
        }
        
        final String rawOwner = extension.getRawOwner();
        if (MacroExpander.containsMacro(rawOwner))
        {
          extension.setOwner( new MetaResourceString(getResourceBundle(context), 
                                                     MacroExpander.stripOffBrackets(rawOwner)) );
        }
        else
        {
          extension.setOwner(rawOwner);  
        }
      }
    };
  }    
  
  
  /**
   * Get the class loader that should be used by default to load an 
   * extension.<p>
   * 
   * This implementation returns Thread.currentThread().getContextClassLoader().
   * 
   * @param extension the extension being processed.
   * @return the classloader to use to load resources for the specified 
   *    extension.
   */
  protected ClassLoader getClassLoader( Extension extension )
  {
    return Thread.currentThread().getContextClassLoader();
  }

  protected void addExtensionSourceToClasspath( ElementContext context )
  {
    // no-op
  }
}
