/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.model;

import java.io.IOException;
import java.io.Reader;

import java.net.URI;

import javax.ide.model.spi.DocumentImpl;
import javax.ide.model.spi.ElementImpl;
import javax.ide.model.spi.ModelAdapterFactory;

/**
 *  The <CODE>Document</CODE> interface is implemented by objects that can
 *  be saved and restored. These objects are generally persisted on some 
 *  file system and are address using an unique {@link URI}.<p>
 *
 *  Documents are generally created using the {@link DocumentFactory}. This 
 *  ensures that there is always one instance of a document pointed to by
 *  the same {@link URI}.<p>
 *
 *  The specification defines following types of documents:<p>
 *
 *  <ul>
 *    <li>{@link javax.ide.model.Project}s: containing user work,<li><p>
 *    <li>{@link javax.ide.model.text.TextDocument}s: encapsulating text based files,<li><p>
 *    <li>{@link javax.ide.model.java.JavaSource}s: encapsulating java files, and <li><p>
 *    <li>{@link javax.ide.model.xml.XMLDocument}s: encapsulating xml files. <li><p>
 *  </ul>
 *
 *  Extension writers that need to introduce custom document classes, should
 *  extend this class.<p>
 *
 *  When a new document class is introduced, the <code>DocumentFactory</code>
 *  must be told how to recognize the new document.  Extension writers tell 
 *  the document factory how to recognize newly introduced document classes 
 *  in the extension deployment descriptor.
 */
public class Document extends Element
{
  private URI _uri;
  private ElementDisplayInfo _displayInfo;

  protected final ElementImpl getElementImpl()
  {
    return 
      ModelAdapterFactory.getModelAdapterFactory().getImpl( this );
  }
  
  private DocumentImpl getDocumentImpl()
  {
    return (DocumentImpl) getElementImpl();
  }
  
  /**
   * Set the display info for this element. This can be used by custom
   * documents to customize their display in the IDE.
   * 
   * @param displayInfo the display info for this document. If null, the
   *    IDE will display this element in whatever the default way is standard
   *    for elements of this type.
   */
  protected final void setDisplayInfo( ElementDisplayInfo displayInfo )
  {
    _displayInfo = displayInfo;
  }
  
  public ElementDisplayInfo getDisplayInfo()
  {
    return _displayInfo;
  }

  /**
   *  Returns the {@link URI} that identifies this
   *  <CODE>Document</CODE>.  Parts of the IDE will use the value of
   *  this {@link URI} as a hash key for caching UI components for this
   *  <CODE>Document</CODE>.  Therefore, {@link URI} uniqueness is
   *  important.
   *
   *  @return The {@link URI} identifying this <CODE>Document</CODE>.
   */
  public final URI getURI()
  {
    return _uri;
  }

  /**
   *  Sets the {@link URI} associated with this <CODE>Document</CODE>.
   *  It is important that the {@link URI} only be changed when the
   *  <CODE>Document</CODE> has just been created or when all caches
   *  keyed on the previous {@link URI} can also be updated.
   *
   *  @param uri The {@link URI} to set.
   */
  public final void setURI( URI uri )
  {
    _uri = uri;
  }

  /**
   *  Returns <CODE>true</CODE> if the object's data has already been
   *  loaded.
   */
  public final boolean isOpen()
  {
    return getDocumentImpl().isOpen();
  }

  /**
   *  Returns <CODE>true</CODE> if the document's data has never been
   *  saved.
   */
  public final boolean isNew()
  {
    return getDocumentImpl().isNew();
  }

  /**
   *  Opens the <CODE>Document</CODE> and loads any associated data
   *  into the appropriate data structures. This method notifies listeners
   *  if the document is successfully opened. 
   *  
   *  @exception IOException if the document cannot be opened.
   */
  public final void open() throws IOException
  {
    getDocumentImpl().open();
  }

  /**
   *  Closes the <CODE>Document</CODE> and unloads any associated data.
   *  When this method returns, the state of the <CODE>Document</CODE>
   *  object should be equivalent to when the <CODE>Document</CODE>
   *  object has just been instantiated but not yet opened.
   *  
   *  @exception IOException if the document cannot be closed.
   */
  public final void close() throws IOException
  {
    getDocumentImpl().close();
  }

  /**
   *  Saves the contents of the document.
   */
  public final void save() throws IOException
  {
    getDocumentImpl().save();
  }

  /**
   *  Returns <CODE>true</CODE> if the document's data has never been
   *  saved.
   *
   *  @return <CODE>true</CODE> if the document's data has never been
   *  saved.
   */
  public final boolean isReadOnly()
  {
    return getDocumentImpl().isReadOnly();
  }

  /**
   *  True if the data in the object has been modified.
   *
   *  @return <CODE>true</CODE> if the data in the object has been modified.
   */
  public final boolean isDirty()
  {
    return getDocumentImpl().isDirty();
  }

  /**
   *  Marks the data with the specified dirty state.  
   *
   *  @param  dirty  If <CODE>true</CODE>, sets the object as being
   *  dirty; if <CODE>false</CODE>, sets the object as being up-to-date.
   */
  public final void markDirty( boolean dirty )
  {
    getDocumentImpl().markDirty( dirty );
  }

  /**
   *  Returns the timestamp associated with the <CODE>Document</CODE>,
   *  which indicates the time at which the document was last modified.
   *  The returned <CODE>long</CODE> is expressed in milliseconds since
   *  00:00:00 GMT Jan 1, 1970.
   *
   *  @return the <CODE>Document</CODE>'s time stamp.
   */
  public final long getTimestamp()
  {
    return getDocumentImpl().getTimestamp();  
  }

  /**
   *  Gets an {@link Reader} that can be used to read the contents
   *  of this object.
   *
   *  @return an {@link Reader}, or <CODE>null</CODE> if the
   *  document has no contents.
   *
   *  @exception IOException if a security manager exists and its
   *             <CODE>checkRead</CODE> method denies read access.
   */
  public final Reader getReader() throws IOException
  {
    return getDocumentImpl().getReader();
  }

  /**
   * Add a {@link DocumentListener} to the listener list.
   * A <code>DocumentEvent</code> will be fired in response modifying
   * the contents of this document.
   */
  public final void addDocumentListener( DocumentListener listener )
  {
    getDocumentImpl().addDocumentListener( listener );
  }

  /**
   * Removes a {@link DocumentListener} from the listener list.
   */
  public final void removeDocumentListener( DocumentListener listener )
  {
    getDocumentImpl().removeDocumentListener( listener );
  }
  
  //--------------------------------------------------------------------------
  // Object overrides.
  //--------------------------------------------------------------------------
  public final int hashCode() 
  {
    return 42 + getURI().hashCode();
  }

  public boolean equals( Object other )
  {
    if ( other == this )
    {
      return true;
    }
    
    if ( !( other instanceof Document) )
    {
      return false;
    }
    
    Document otherDocument = (Document) other;

    return  otherDocument.getURI().equals( getURI() );
  }
}
