/*
 * Decompiled with CFR 0.152.
 */
package io.grpc.internal;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.base.Stopwatch;
import com.google.common.base.Supplier;
import com.google.common.collect.ImmutableMap;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import io.grpc.Metadata;
import io.grpc.Status;
import io.grpc.internal.SharedResourceHolder;
import io.grpc.internal.WithLogId;
import java.lang.reflect.Method;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import javax.annotation.Nullable;

public final class GrpcUtil {
    public static final boolean IS_RESTRICTED_APPENGINE = "Production".equals(System.getProperty("com.google.appengine.runtime.environment")) && "1.7".equals(System.getProperty("java.specification.version"));
    public static final Metadata.Key<Long> TIMEOUT_KEY = Metadata.Key.of("grpc-timeout", new TimeoutMarshaller());
    public static final Metadata.Key<String> MESSAGE_ENCODING_KEY = Metadata.Key.of("grpc-encoding", Metadata.ASCII_STRING_MARSHALLER);
    public static final Metadata.Key<String> MESSAGE_ACCEPT_ENCODING_KEY = Metadata.Key.of("grpc-accept-encoding", Metadata.ASCII_STRING_MARSHALLER);
    public static final Metadata.Key<String> CONTENT_TYPE_KEY = Metadata.Key.of("content-type", Metadata.ASCII_STRING_MARSHALLER);
    public static final Metadata.Key<String> USER_AGENT_KEY = Metadata.Key.of("user-agent", Metadata.ASCII_STRING_MARSHALLER);
    public static final int DEFAULT_PORT_PLAINTEXT = 80;
    public static final int DEFAULT_PORT_SSL = 443;
    public static final String CONTENT_TYPE_GRPC = "application/grpc";
    public static final String HTTP_METHOD = "POST";
    public static final String TE_TRAILERS = "trailers";
    public static final String TIMEOUT = "grpc-timeout";
    public static final String MESSAGE_ENCODING = "grpc-encoding";
    public static final String MESSAGE_ACCEPT_ENCODING = "grpc-accept-encoding";
    public static final int DEFAULT_MAX_MESSAGE_SIZE = 0x400000;
    public static final int DEFAULT_MAX_HEADER_LIST_SIZE = 8192;
    public static final Splitter ACCEPT_ENCODING_SPLITER = Splitter.on((char)',').trimResults();
    private static final String IMPLEMENTATION_VERION = GrpcUtil.getImplementationVersion();
    public static final long DEFAULT_KEEPALIVE_DELAY_NANOS = TimeUnit.MINUTES.toNanos(1L);
    public static final long DEFAULT_KEEPALIVE_TIMEOUT_NANOS = TimeUnit.MINUTES.toNanos(2L);
    public static final SharedResourceHolder.Resource<ExecutorService> SHARED_CHANNEL_EXECUTOR = new SharedResourceHolder.Resource<ExecutorService>(){
        private static final String name = "grpc-default-executor";

        @Override
        public ExecutorService create() {
            return Executors.newCachedThreadPool(GrpcUtil.getThreadFactory("grpc-default-executor-%d", true));
        }

        @Override
        public void close(ExecutorService instance) {
            instance.shutdown();
        }

        public String toString() {
            return name;
        }
    };
    public static final SharedResourceHolder.Resource<ScheduledExecutorService> TIMER_SERVICE = new SharedResourceHolder.Resource<ScheduledExecutorService>(){

        @Override
        public ScheduledExecutorService create() {
            ScheduledExecutorService service = Executors.newScheduledThreadPool(1, GrpcUtil.getThreadFactory("grpc-timer-%d", true));
            try {
                Method method = service.getClass().getMethod("setRemoveOnCancelPolicy", Boolean.TYPE);
                method.invoke((Object)service, true);
            }
            catch (NoSuchMethodException method) {
            }
            catch (RuntimeException e) {
                throw e;
            }
            catch (Exception e) {
                throw new RuntimeException(e);
            }
            return service;
        }

        @Override
        public void close(ScheduledExecutorService instance) {
            instance.shutdown();
        }
    };
    static final Supplier<Stopwatch> STOPWATCH_SUPPLIER = new Supplier<Stopwatch>(){

        public Stopwatch get() {
            return Stopwatch.createUnstarted();
        }
    };

    public static Status httpStatusToGrpcStatus(int httpStatusCode) {
        switch (httpStatusCode) {
            case 401: {
                return Status.UNAUTHENTICATED;
            }
            case 403: {
                return Status.PERMISSION_DENIED;
            }
        }
        if (httpStatusCode < 100) {
            return Status.UNKNOWN;
        }
        if (httpStatusCode < 200) {
            return Status.INTERNAL;
        }
        if (httpStatusCode < 300) {
            return Status.UNKNOWN;
        }
        return Status.UNKNOWN;
    }

    public static boolean isGrpcContentType(String contentType) {
        if (contentType == null) {
            return false;
        }
        if (CONTENT_TYPE_GRPC.length() > contentType.length()) {
            return false;
        }
        if (!(contentType = contentType.toLowerCase()).startsWith(CONTENT_TYPE_GRPC)) {
            return false;
        }
        if (contentType.length() == CONTENT_TYPE_GRPC.length()) {
            return true;
        }
        char nextChar = contentType.charAt(CONTENT_TYPE_GRPC.length());
        return nextChar == '+' || nextChar == ';';
    }

    public static String getGrpcUserAgent(String transportName, @Nullable String applicationUserAgent) {
        StringBuilder builder = new StringBuilder();
        if (applicationUserAgent != null) {
            builder.append(applicationUserAgent);
            builder.append(' ');
        }
        builder.append("grpc-java-");
        builder.append(transportName);
        builder.append(IMPLEMENTATION_VERION);
        return builder.toString();
    }

    public static URI authorityToUri(String authority) {
        URI uri;
        Preconditions.checkNotNull((Object)authority, (Object)"authority");
        try {
            uri = new URI(null, authority, null, null, null);
        }
        catch (URISyntaxException ex) {
            throw new IllegalArgumentException("Invalid authority: " + authority, ex);
        }
        return uri;
    }

    public static String checkAuthority(String authority) {
        URI uri = GrpcUtil.authorityToUri(authority);
        Preconditions.checkArgument((uri.getHost() != null ? 1 : 0) != 0, (String)"No host in authority '%s'", (Object[])new Object[]{authority});
        Preconditions.checkArgument((uri.getUserInfo() == null ? 1 : 0) != 0, (String)"Userinfo must not be present on authority: '%s'", (Object[])new Object[]{authority});
        return authority;
    }

    public static String authorityFromHostAndPort(String host, int port) {
        try {
            return new URI(null, null, host, port, null, null, null).getAuthority();
        }
        catch (URISyntaxException ex) {
            throw new IllegalArgumentException("Invalid host or port: " + host + " " + port, ex);
        }
    }

    public static ThreadFactory getThreadFactory(String nameFormat, boolean daemon) {
        ThreadFactory threadFactory = MoreExecutors.platformThreadFactory();
        if (IS_RESTRICTED_APPENGINE) {
            return threadFactory;
        }
        return new ThreadFactoryBuilder().setThreadFactory(threadFactory).setDaemon(daemon).setNameFormat(nameFormat).build();
    }

    public static String getLogId(WithLogId subject) {
        return subject.getClass().getSimpleName() + "@" + Integer.toHexString(subject.hashCode());
    }

    private GrpcUtil() {
    }

    private static String getImplementationVersion() {
        String version = GrpcUtil.class.getPackage().getImplementationVersion();
        if (version != null) {
            return "/" + version;
        }
        return "";
    }

    @VisibleForTesting
    static class TimeoutMarshaller
    implements Metadata.AsciiMarshaller<Long> {
        private static final ImmutableMap<Character, TimeUnit> UNITS = ImmutableMap.builder().put((Object)Character.valueOf('n'), (Object)TimeUnit.NANOSECONDS).put((Object)Character.valueOf('u'), (Object)TimeUnit.MICROSECONDS).put((Object)Character.valueOf('m'), (Object)TimeUnit.MILLISECONDS).put((Object)Character.valueOf('S'), (Object)TimeUnit.SECONDS).put((Object)Character.valueOf('M'), (Object)TimeUnit.MINUTES).put((Object)Character.valueOf('H'), (Object)TimeUnit.HOURS).build();

        TimeoutMarshaller() {
        }

        @Override
        public String toAsciiString(Long timeoutNanos) {
            Preconditions.checkArgument((timeoutNanos >= 0L ? 1 : 0) != 0, (Object)"Negative timeout");
            int cutoff = 100000000;
            for (Map.Entry unit : UNITS.entrySet()) {
                long timeout = ((TimeUnit)((Object)unit.getValue())).convert(timeoutNanos, TimeUnit.NANOSECONDS);
                if (timeout >= (long)cutoff) continue;
                return Long.toString(timeout) + unit.getKey();
            }
            throw new IllegalArgumentException("Timeout too large");
        }

        @Override
        public Long parseAsciiString(String serialized) {
            Preconditions.checkArgument((serialized.length() > 0 ? 1 : 0) != 0, (Object)"empty timeout");
            Preconditions.checkArgument((serialized.length() <= 9 ? 1 : 0) != 0, (Object)"bad timeout format");
            String valuePart = serialized.substring(0, serialized.length() - 1);
            char unit = serialized.charAt(serialized.length() - 1);
            TimeUnit timeUnit = (TimeUnit)((Object)UNITS.get((Object)Character.valueOf(unit)));
            if (timeUnit != null) {
                return timeUnit.toNanos(Long.parseLong(valuePart));
            }
            throw new IllegalArgumentException(String.format("Invalid timeout unit: %s", Character.valueOf(unit)));
        }
    }

    public static enum Http2Error {
        NO_ERROR(0, Status.UNAVAILABLE),
        PROTOCOL_ERROR(1, Status.INTERNAL),
        INTERNAL_ERROR(2, Status.INTERNAL),
        FLOW_CONTROL_ERROR(3, Status.INTERNAL),
        SETTINGS_TIMEOUT(4, Status.INTERNAL),
        STREAM_CLOSED(5, Status.INTERNAL),
        FRAME_SIZE_ERROR(6, Status.INTERNAL),
        REFUSED_STREAM(7, Status.UNAVAILABLE),
        CANCEL(8, Status.CANCELLED),
        COMPRESSION_ERROR(9, Status.INTERNAL),
        CONNECT_ERROR(10, Status.INTERNAL),
        ENHANCE_YOUR_CALM(11, Status.RESOURCE_EXHAUSTED.withDescription("Bandwidth exhausted")),
        INADEQUATE_SECURITY(12, Status.PERMISSION_DENIED.withDescription("Permission denied as protocol is not secure enough to call")),
        HTTP_1_1_REQUIRED(13, Status.UNKNOWN);

        private static final Http2Error[] codeMap;
        private final int code;
        private final Status status;

        private static Http2Error[] buildHttp2CodeMap() {
            Http2Error[] errors = Http2Error.values();
            int size = (int)errors[errors.length - 1].code() + 1;
            Http2Error[] http2CodeMap = new Http2Error[size];
            for (Http2Error error : errors) {
                int index = (int)error.code();
                http2CodeMap[index] = error;
            }
            return http2CodeMap;
        }

        private Http2Error(int code, Status status) {
            this.code = code;
            this.status = status.augmentDescription("HTTP/2 error code: " + this.name());
        }

        public long code() {
            return this.code;
        }

        public Status status() {
            return this.status;
        }

        public static Http2Error forCode(long code) {
            if (code >= (long)codeMap.length || code < 0L) {
                return null;
            }
            return codeMap[(int)code];
        }

        public static Status statusForCode(long code) {
            Http2Error error = Http2Error.forCode(code);
            if (error == null) {
                Status.Code statusCode = INTERNAL_ERROR.status().getCode();
                return Status.fromCodeValue(statusCode.value()).withDescription("Unrecognized HTTP/2 error code: " + code);
            }
            return error.status();
        }

        static {
            codeMap = Http2Error.buildHttp2CodeMap();
        }
    }
}

