/*
 * Decompiled with CFR 0.152.
 */
package git4idea.update;

import com.intellij.dvcs.MultiRootMessage;
import com.intellij.openapi.diagnostic.Logger;
import com.intellij.openapi.progress.ProgressIndicator;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.util.Key;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.openapi.vcs.VcsNotifier;
import com.intellij.openapi.vfs.VirtualFile;
import com.intellij.util.ArrayUtil;
import git4idea.GitLocalBranch;
import git4idea.GitRemoteBranch;
import git4idea.GitUtil;
import git4idea.GitVcs;
import git4idea.branch.GitBranchUtil;
import git4idea.commands.Git;
import git4idea.commands.GitCommandResult;
import git4idea.commands.GitLineHandlerAdapter;
import git4idea.repo.GitBranchTrackInfo;
import git4idea.repo.GitRemote;
import git4idea.repo.GitRepository;
import git4idea.repo.GitRepositoryManager;
import git4idea.update.GitFetchResult;
import git4idea.util.GitUIUtil;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public class GitFetcher {
    private static final Logger LOG = Logger.getInstance(GitFetcher.class);
    private final Project myProject;
    private final GitRepositoryManager myRepositoryManager;
    private final ProgressIndicator myProgressIndicator;
    private final boolean myFetchAll;
    private final GitVcs myVcs;
    private final Collection<Exception> myErrors = new ArrayList<Exception>();

    public GitFetcher(@NotNull Project project, @NotNull ProgressIndicator progressIndicator, boolean fetchAll) {
        this.myProject = project;
        this.myProgressIndicator = progressIndicator;
        this.myFetchAll = fetchAll;
        this.myRepositoryManager = GitUtil.getRepositoryManager(this.myProject);
        this.myVcs = GitVcs.getInstance(project);
    }

    public GitFetchResult fetch(@NotNull GitRepository repository) {
        GitFetchResult fetchResult = this.myFetchAll ? GitFetcher.fetchAll(repository) : this.fetchCurrentRemote(repository);
        repository.getRepositoryFiles().refresh(false);
        return fetchResult;
    }

    @NotNull
    public GitFetchResult fetch(@NotNull VirtualFile root, @NotNull String remoteName, @Nullable String branch) {
        GitRepository repository = (GitRepository)this.myRepositoryManager.getRepositoryForRoot(root);
        if (repository == null) {
            return GitFetcher.logError("Repository can't be null for " + root, ((Object)((Object)this.myRepositoryManager)).toString());
        }
        GitRemote remote = GitUtil.findRemoteByName(repository, remoteName);
        if (remote == null) {
            return GitFetcher.logError("Couldn't find remote with the name " + remoteName, null);
        }
        return this.fetchRemote(repository, remote, branch);
    }

    private static GitFetchResult logError(@NotNull String message, @Nullable String additionalInfo) {
        String addInfo = additionalInfo != null ? "\n" + additionalInfo : "";
        LOG.error(message + addInfo);
        return GitFetchResult.error(message);
    }

    @NotNull
    private GitFetchResult fetchCurrentRemote(@NotNull GitRepository repository) {
        FetchParams fetchParams = GitFetcher.getFetchParams(repository);
        if (fetchParams.isError()) {
            return fetchParams.getError();
        }
        GitRemote remote = fetchParams.getRemote();
        return this.fetchRemote(repository, remote, null);
    }

    @NotNull
    private GitFetchResult fetchRemote(@NotNull GitRepository repository, @NotNull GitRemote remote, @Nullable String branch) {
        return GitFetcher.fetchNatively(repository, remote, branch);
    }

    @NotNull
    private GitFetchResult fetchCurrentBranch(@NotNull GitRepository repository) {
        FetchParams fetchParams = GitFetcher.getFetchParams(repository);
        if (fetchParams.isError()) {
            return fetchParams.getError();
        }
        GitRemote remote = fetchParams.getRemote();
        String remoteBranch = fetchParams.getRemoteBranch().getNameForRemoteOperations();
        return GitFetcher.fetchNatively(repository, remote, remoteBranch);
    }

    @NotNull
    private static FetchParams getFetchParams(@NotNull GitRepository repository) {
        GitLocalBranch currentBranch = repository.getCurrentBranch();
        if (currentBranch == null) {
            String message = "Current branch can't be null here. \nRepository: " + repository;
            LOG.error(message);
            return new FetchParams(GitFetchResult.error(new Exception(message)));
        }
        GitBranchTrackInfo trackInfo = GitBranchUtil.getTrackInfoForBranch(repository, currentBranch);
        if (trackInfo == null) {
            String message = "Tracked info is null for branch " + currentBranch + "\n Repository: " + repository;
            LOG.error(message);
            return new FetchParams(GitFetchResult.error(new Exception(message)));
        }
        GitRemote remote = trackInfo.getRemote();
        return new FetchParams(remote, trackInfo.getRemoteBranch());
    }

    @NotNull
    private static GitFetchResult fetchAll(@NotNull GitRepository repository) {
        GitFetchResult fetchResult = GitFetchResult.success();
        for (GitRemote remote : repository.getRemotes()) {
            String url = remote.getFirstUrl();
            if (url == null) {
                LOG.error("URL is null for remote " + remote.getName());
                continue;
            }
            GitFetchResult res = GitFetcher.fetchNatively(repository, remote, null);
            res.addPruneInfo(fetchResult.getPrunedRefs());
            fetchResult = res;
            if (fetchResult.isSuccess()) continue;
            break;
        }
        return fetchResult;
    }

    @NotNull
    private static GitFetchResult fetchNatively(@NotNull GitRepository repository, @NotNull GitRemote remote, @Nullable String branch) {
        String[] stringArray;
        Git git = Git.getInstance();
        if (branch != null) {
            String[] stringArray2 = new String[1];
            stringArray = stringArray2;
            stringArray2[0] = GitFetcher.getFetchSpecForBranch(branch, remote.getName());
        } else {
            stringArray = ArrayUtil.EMPTY_STRING_ARRAY;
        }
        String[] additionalParams = stringArray;
        GitFetchPruneDetector pruneDetector = new GitFetchPruneDetector();
        GitCommandResult result = git.fetch(repository, remote, Collections.singletonList(pruneDetector), additionalParams);
        GitFetchResult fetchResult = result.success() ? GitFetchResult.success() : (result.cancelled() ? GitFetchResult.cancel() : GitFetchResult.error(result.getErrorOutputAsJoinedString()));
        fetchResult.addPruneInfo(pruneDetector.getPrunedRefs());
        return fetchResult;
    }

    private static String getRidOfPrefixIfExists(String branch) {
        if (branch.startsWith("refs/heads/")) {
            return branch.substring("refs/heads/".length());
        }
        return branch;
    }

    @NotNull
    public static String getFetchSpecForBranch(@NotNull String branch, @NotNull String remoteName) {
        branch = GitFetcher.getRidOfPrefixIfExists(branch);
        return "refs/heads/" + branch + ":" + "refs/remotes/" + remoteName + "/" + branch;
    }

    @NotNull
    public Collection<Exception> getErrors() {
        return this.myErrors;
    }

    public static void displayFetchResult(@NotNull Project project, @NotNull GitFetchResult result, @Nullable String errorNotificationTitle, @NotNull Collection<? extends Exception> errors) {
        if (result.isSuccess()) {
            VcsNotifier.getInstance((Project)project).notifySuccess("Fetched successfully" + result.getAdditionalInfo());
        } else if (result.isCancelled()) {
            VcsNotifier.getInstance((Project)project).notifyMinorWarning("", "Fetch cancelled by user" + result.getAdditionalInfo());
        } else if (result.isNotAuthorized()) {
            String description;
            String title;
            if (errorNotificationTitle != null) {
                title = errorNotificationTitle;
                description = "Fetch failed: couldn't authorize";
            } else {
                title = "Fetch failed";
                description = "Couldn't authorize";
            }
            description = description + result.getAdditionalInfo();
            GitUIUtil.notifyMessage(project, title, description, true, null);
        } else {
            GitVcs instance = GitVcs.getInstance(project);
            if (instance != null && instance.getExecutableValidator().isExecutableValid()) {
                GitUIUtil.notifyMessage(project, "Fetch failed", result.getAdditionalInfo(), true, errors);
            }
        }
    }

    public boolean fetchRootsAndNotify(@NotNull Collection<GitRepository> roots, @Nullable String errorNotificationTitle, boolean notifySuccess) {
        MultiRootMessage additionalInfo = new MultiRootMessage(this.myProject, GitUtil.getRootsFromRepositories(roots), true);
        for (GitRepository repository : roots) {
            LOG.info("fetching " + repository);
            GitFetchResult result = this.fetch(repository);
            String ai = result.getAdditionalInfo();
            if (!StringUtil.isEmptyOrSpaces((String)ai)) {
                additionalInfo.append(repository.getRoot(), ai);
            }
            if (result.isSuccess()) continue;
            ArrayList<Exception> errors = new ArrayList<Exception>(this.getErrors());
            errors.addAll(result.getErrors());
            GitFetcher.displayFetchResult(this.myProject, result, errorNotificationTitle, errors);
            return false;
        }
        if (notifySuccess) {
            VcsNotifier.getInstance((Project)this.myProject).notifySuccess("Fetched successfully");
        }
        if (!additionalInfo.asString().isEmpty()) {
            VcsNotifier.getInstance((Project)this.myProject).notifyMinorInfo("Fetch details", additionalInfo.asString());
        }
        return true;
    }

    private static class FetchParams {
        private GitRemote myRemote;
        private GitRemoteBranch myRemoteBranch;
        private GitFetchResult myError;

        FetchParams(GitFetchResult error) {
            this.myError = error;
        }

        FetchParams(GitRemote remote, GitRemoteBranch remoteBranch) {
            this.myRemote = remote;
            this.myRemoteBranch = remoteBranch;
        }

        boolean isError() {
            return this.myError != null;
        }

        public GitFetchResult getError() {
            return this.myError;
        }

        public GitRemote getRemote() {
            return this.myRemote;
        }

        public GitRemoteBranch getRemoteBranch() {
            return this.myRemoteBranch;
        }
    }

    private static class GitFetchPruneDetector
    extends GitLineHandlerAdapter {
        private static final Pattern PRUNE_PATTERN = Pattern.compile("\\s*x\\s*\\[deleted\\].*->\\s*(\\S*)");
        @NotNull
        private final Collection<String> myPrunedRefs = new ArrayList<String>();

        private GitFetchPruneDetector() {
        }

        @Override
        public void onLineAvailable(String line, Key outputType) {
            Matcher matcher = PRUNE_PATTERN.matcher(line);
            if (matcher.matches()) {
                this.myPrunedRefs.add(matcher.group(1));
            }
        }

        @NotNull
        public Collection<String> getPrunedRefs() {
            return this.myPrunedRefs;
        }
    }
}

